<?php


require_once __DIR__ . "/IBSngUtils.php";

class IBSngRequests
{
    static public IBSngRequests $_instance;
    protected string $hostname, $username, $password;
    protected int $port, $timeout;
    protected bool $isConnected = FALSE;
    protected array $loginData = [];
    protected bool $autoConnect = TRUE;
    protected ?string $cookiePathName = NULL;
    protected $handler = NULL;
    private IBSngUtils $IBSngUtils;
    protected string $agent = 'phpIBSng web Api';


    public function setAutoConnect($autoConnect): void
    {
        $this->autoConnect = $autoConnect;
    }

    public function __construct(array $loginArray)
    {
        $time = microtime(true);

        if (!extension_loaded('curl')) {
            throw new Exception ("You need to load/activate the curl extension.");
        }
        libxml_use_internal_errors(TRUE);

        self::$_instance = $this;

        $this->loginData = $loginArray;
        if (!$this->loginData['username'] ||
            !$this->loginData['password'] ||
            !$this->loginData['hostname']
        ) {

            throw new Exception('IBSng needs correct login information');
        }

        $this->hostname = $loginArray['hostname'];
        $this->username = $loginArray['username'];
        $this->password = $loginArray['password'];
        $this->port = $loginArray['port'] ?? 80;
        $this->timeout = $loginArray['timeout'] ?? 30;

        $this->cookiePathName = sys_get_temp_dir() . '/.' . self::class;

        if ($this->autoConnect) {
            $this->connect($time);
            echo "0 : " . microtime(true) - $time . "<br>";

            $this->IBSngUtils = new IBSngUtils();
            echo "00 : " . microtime(true) - $time . "<br>";

        }

    }
    private function initCurlHandle(): void {
        if ($this->handler) return;
        $this->handler = curl_init();
        curl_setopt($this->handler, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($this->handler, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($this->handler, CURLOPT_MAXREDIRS, 3);
        curl_setopt($this->handler, CURLOPT_NOSIGNAL, true);              // faster timeouts in PHP
        curl_setopt($this->handler, CURLOPT_CONNECTTIMEOUT, 5);           // was 0 (infinite)
        curl_setopt($this->handler, CURLOPT_TIMEOUT, 20);                 // overall cap
        curl_setopt($this->handler, CURLOPT_LOW_SPEED_LIMIT, 1024);       // 1KB/s …
        curl_setopt($this->handler, CURLOPT_LOW_SPEED_TIME, 10);          // … for 10s -> abort
        curl_setopt($this->handler, CURLOPT_ENCODING, "");                // accept gzip/deflate
        curl_setopt($this->handler, CURLOPT_IPRESOLVE, CURL_IPRESOLVE_V4);// skip slow AAAA (optional)
        curl_setopt($this->handler, CURLOPT_USERAGENT, $this->agent ?? 'curl/php');
        curl_setopt($this->handler, CURLOPT_COOKIEJAR, __DIR__ . '/cookie.txt');
        curl_setopt($this->handler, CURLOPT_COOKIEFILE, __DIR__ . '/cookie.txt');
        // Keep connections alive
        curl_setopt($this->handler, CURLOPT_FORBID_REUSE, false);
        curl_setopt($this->handler, CURLOPT_FRESH_CONNECT, false);
        // If supported by your libcurl:
        if (defined('CURLOPT_TCP_KEEPALIVE')) {
            curl_setopt($this->handler, CURLOPT_TCP_KEEPALIVE, 1);
            curl_setopt($this->handler, CURLOPT_TCP_KEEPIDLE, 30);
            curl_setopt($this->handler, CURLOPT_TCP_KEEPINTVL, 15);
        }
    }
    protected function request($action, $postData = array(), $header = []): bool|string
    {
//        if (empty($action)) {
//            throw new Exception('Url specified in request is empty');
//        }
//
//        $url = 'http://' . $this->hostname . '/' . $action;
//        $cookieFile = __DIR__ . '/cookie.txt'; // Define the path to the cookie file
//
//        // Initialize cURL session
//        $this->handler = curl_init();
//        curl_setopt($this->handler, CURLOPT_CONNECTTIMEOUT, 0);
//        curl_setopt($this->handler, CURLOPT_TIMEOUT, $this->timeout);
//        curl_setopt($this->handler, CURLOPT_URL, $url);
//        curl_setopt($this->handler, CURLOPT_PORT, $this->port);
//        curl_setopt($this->handler, CURLOPT_POST, TRUE);
//        curl_setopt($this->handler, CURLOPT_POSTFIELDS, $postData);
//        curl_setopt($this->handler, CURLOPT_HEADER, $header);
//        curl_setopt($this->handler, CURLOPT_RETURNTRANSFER, TRUE);
//        curl_setopt($this->handler, CURLOPT_FOLLOWLOCATION, TRUE);
//        curl_setopt($this->handler, CURLOPT_USERAGENT, $this->agent);
//        curl_setopt($this->handler, CURLOPT_SSL_VERIFYHOST, FALSE);
//        curl_setopt($this->handler, CURLOPT_SSL_VERIFYPEER, FALSE);
//
//        // Use the cookie from the previous login if it exists
//        if (file_exists($cookieFile)) {
//            curl_setopt($this->handler, CURLOPT_COOKIEFILE, $cookieFile); // Use stored cookie
//        }
//
//        // Save cookies for future requests
//        curl_setopt($this->handler, CURLOPT_COOKIEJAR, $cookieFile); // Save cookies to file
//
//        // Execute the request
//        $output = curl_exec($this->handler);
//
//        // Handle errors
//        if (curl_errno($this->handler) != 0) {
//            throw new Exception(curl_error($this->handler) . ' ' . $url . "\n");
//        }
//
//        curl_close($this->handler);
//        return $output;

        if (empty($action)) {
            throw new \Exception('Url specified in request is empty');
        }
        $this->initCurlHandle();

        $url = 'http://' . $this->hostname . '/' . ltrim($action, '/');

        // Build headers: IMPORTANT: use CURLOPT_HTTPHEADER (not CURLOPT_HEADER)
        $headers = array_merge([
            'Connection: keep-alive',
            'Accept: */*',
            'Expect:',                         // disable 100-continue delay
        ]);

        // Normalize body
        if (is_array($postData)) {
            $body = http_build_query($postData);   // small & fast x-www-form-urlencoded
            $headers[] = 'Content-Type: application/x-www-form-urlencoded';
        } else {
            $body = (string)$postData;             // already-serialized body
            if (!preg_grep('/^Content-Type:/i', $headers)) {
                $headers[] = 'Content-Type: application/octet-stream';
            }
        }

        // If you know the backend IP, pin it to skip DNS (optional but fast):
        // curl_setopt($this->handler, CURLOPT_RESOLVE, ["{$this->hostname}:{$this->port}:203.0.113.10"]);

        curl_setopt_array($this->handler, [
            CURLOPT_URL         => $url,
            CURLOPT_PORT        => $this->port,
            CURLOPT_CUSTOMREQUEST=> 'POST',
            CURLOPT_POSTFIELDS  => $body,
            CURLOPT_HTTPHEADER  => $headers,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_SSL_VERIFYPEER => false,
            // If your server supports HTTP/2 only on HTTPS, switch to https and:
            // CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_2TLS,
        ]);

        $output = curl_exec($this->handler);

        if ($output === false) {
            $err = curl_error($this->handler);
            $code = curl_errno($this->handler);
            throw new \Exception("cURL error ($code): $err $url");
        }
        return $output;

    }

    public function editUser($userName, $attribute, $value, $userPasswordEdit)
    {
        return $this->_editUser($userName, $attribute, $value, $userPasswordEdit);
    }

    public function getAllUsersList()
    {
        return $this->_getAllUsersList();
    }

    public function changeCredit($userId, $credit, $comment = "")
    {
        return $this->_changeCredit($userId, $credit);
    }

    public function addDays($userName, $days)
    {
        return $this->_addDays($userName, $days);
    }

    public function addMultiLogin($userName, $multiLogin)
    {
        return $this->_addMultiLogin($userName,$multiLogin);
    }

    protected function hostNameHealth($hostname = FALSE, $port = FALSE)
    {
        if (!$hostname) {
            $hostname = $this->hostname;
        }
        if (!$port) {
            $port = $this->port;
        }
        $fp = @fsockopen($hostname, $port);
        return $fp;
    }

    protected function getCookie(): ?string
    {
        return $this->cookiePathName;
    }

    public function connect($time): true|array
    {
        if ($this->isConnected()) {
            return ['result' => TRUE];
        }
        if (!$this->login()['result']) {
            return ['result' => FALSE, 'error' => "Can't login to IBSng. Wrong username or password"];
        }
        return $this->isConnected = TRUE;
    }

    public function disconnect(): void
    {
        if ($this->handler) {
            @unlink($this->getCookie());
            @curl_close($this->handler);
        }
    }

    public function isConnected(): bool
    {
        return $this->isConnected;
    }

    public function addUser(string $username, string $password, string $group, int $credit, $multiLogin = NULL)
    {
        return $this->_addUser($username, $password, $group, $credit, $multiLogin);
    }

    public function deleteUser($username, $isUid = FALSE): array
    {
        return $this->_delUser($username, $isUid);
    }

    public function resetFirstLogin($userName)
    {
        return $this->_resetFirstLogin($userName);
    }

    public function getUser($username, $withPassword = TRUE, $isUid = FALSE): array
    {
        return $this->getUserInfo($username, $withPassword, $isUid);
    }

    /**
     * if $type set to custom group expire date will be ignored and add custom days to
     * user expire date with consideration user's remained time
     * is TESTED : Have first login and expire date > time()
     * is TESTED : didn't have first login and expire date
     * MUST be TESTED : Have first login and expire date < time()
     */
    public function reNewUser(string $username, $serviceInfo, string $type = "", int $customDays = 0)
    {
        $userInfo = $this->getUserInfo($username);
        if (isset($userInfo['info'])) {
            return $this->_reNewUser($userInfo['info'], $serviceInfo, $type, $customDays);
        }
        else {
            return ['result' => FALSE, "error" => "User not found"];
        }
    }

    public function updateChargeUser(string $username, string $Charge): array
    {
        return $this->_updateChargeUser($username, $Charge);
    }

    public function addChargeGroup($ras, $ip, $day = 31): array
    {
        return $this->_addChargeGroup($ras, $ip, $day);
    }

    public function cleanReport(string $log, int $time, string $time_value = 'Days'): array
    {
        $log2 = str_replace(['connection_logs', 'credit_changes', 'user_audit_logs', 'snapshots', 'web_analyzer'], ['connection_log', 'credit_change', 'user_audit_log', 'snapshots', 'web_analyzer'], $log);
        return $this->_cleanReport($log, $log2, $time, $time_value);
    }

    protected function login()
    {
        $action = 'IBSng/admin/';
        $postData['username'] = $this->username;
        $postData['password'] = $this->password;
        $output = $this->request($action, $postData, TRUE);
        if (strpos($output, 'admin_index') > 0) {
            return ['result' => TRUE];
        }
        else {
            return ['result' => FALSE];
        }
    }

    protected function _reNewUser($user, $serviceDetails, $type, $customDays)
    {
        $groupName = $serviceDetails['ibsng_group_name'];
        $credit = $serviceDetails['credit'];
        $this->changeCredit($user['uid'],$credit);
        $action = 'IBSng/admin/plugins/edit.php';
        if ($type != "custom") {
            $groupRelativeExpireDate = $this->getGroupsDetails($groupName)['data']['rel_exp_combined'];
            $customDays = $this->IBSngUtils->calculateRelativeExpireDate($groupRelativeExpireDate);
        }
//        $this->addDays($user['username'], $customDays);
        if ($user['first_login'] == '0') {
            if ($user['relative_expire_date'] != "0") {
                $userRelativeDays = $this->IBSngUtils->calculateRelativeExpireDate($user['relative_expire_date']);
            }
            else {
                $userRelativeDays = $this->IBSngUtils->calculateRelativeExpireDate($user['relative_expire_date_gp']);
            }
            $new = $customDays + $userRelativeDays;
        }
        else {
            $exp = $user['nearest_expire_date'];
            if (strtotime($exp) <= time()) {
                $new = $customDays;
            }
            else {
                $date1 = strtotime($exp);
                $remain = $date1 - time();
                $remainDays = round($remain / (3600 * 24));
                $new = $remainDays + $customDays;
            }
            $this->_resetFirstLogin($user['username']);
        }
        echo "NEW : " . $new . "\n";
        $post_data = $this->IBSngUtils->relativeExpireDatePost($user['uid'], $new);
        $output = $this->request($action, $post_data, TRUE);
        return ['result' => TRUE];
    }

    protected function _addChargeGroup($ras, $ip, $day)
    {
        $ip1 = str_replace('.', '!', $ip) . '_ALL_';
        $action = 'IBSng/admin/charge/add_new_charge.php';
        $post_data = [
            'charge_name' => $ras,
            'charge_type' => 'Internet',
            'visible_to_all' => 1,
            'comment' => '',
            'x' => 32,
            'y' => 14,
        ];
        $out = $this->request($action, $post_data, TRUE);
        $action = 'IBSng/admin/charge/add_internet_charge_rule.php?charge_name=' . $ras;
        $post_data = [
            'charge_name' => $ras,
            'charge_rule_id' => 'N/A',
            'rule_start' => '00:00:00',
            'rule_end' => '23:59:59',
            'cpm' => 0,
            'cpk' => 0,
            'assumed_kps' => '99999999',
            'bandwidth_limit_kbytes' => '0',
            'tx_leaf_name' => '',
            'rx_leaf_name' => '',
            'checkall' => 1,
            'Monday' => 1,
            'Tuesday' => 1,
            'Wednesday' => 1,
            'Thursday' => 1,
            'Friday' => 1,
            'Saturday' => 1,
            'Sunday' => 1,
            'ras' => $ip,
            $ip1 => 1,
            'x' => 19,
            'y' => 9,
        ];
        $out = $this->request($action, $post_data);
        $action = 'IBSng/admin/group/add_new_group.php';
        $post_data = [
            'group_name' => $ras,
            'comment' => '',
            'x' => 14,
            'y' => 13,
        ];
        $out = $this->request($action, $post_data);
        $action = 'IBSng/admin/plugins/edit.php';
        $post_data = [
            'target' => 'group',
            'target_id' => $ras,
            'edit_tpl_cs' => 'normal_charge,rel_exp_date',
            'tab1_selected' => 'Exp_Dates',
            'attr_update_method_0' => 'normalCharge',
            'has_normal_charge' => 't',
            'normal_charge' => $ras,
            'attr_update_method_1' => 'relExpDate',
            'has_rel_exp' => 't',
            'rel_exp_date' => $day,
            'rel_exp_date_unit' => 'Days',
            'update' => 1,
        ];
        $out = $this->request($action, $post_data);
        return ['result' => TRUE];
    }

    protected function _updateChargeUser($username, $Charge)
    {
        $action = 'IBSng/admin/plugins/edit.php';
        $user = $this->getUserInfo($username)['info'];

        $post_data = $this->IBSngUtils->userTargetPostData($user['uid']);
        $post_data['edit_tpl_cs'] = 'normal_charge';
        $post_data['attr_update_method_0'] = 'normalCharge';
        $post_data['has_normal_charge'] = 't';
        $post_data['normal_charge'] = $Charge;
        $output = $this->request($action, $post_data, TRUE);
        return ['result' => TRUE];
    }

    protected function _cleanReport($log, $log2, $time, $time_value)
    {
        $action = 'IBSng/admin/report/clean_reports.php';

        $post_data['delete_' . $log] = 'user';
        $post_data[$log2 . '_date'] = $time;
        $post_data[$log2 . '_unit'] = $time_value;
        $output = $this->request($action, $post_data);
        return ['result' => TRUE];
    }

    protected function _addUser($username, $password, $group_name, $credit, $multiLogin = NULL)
    {
        if (!$this->IBSngUtils->validateUserName($username)) {
            return ['result' => FALSE, 'error' => "Bad username"];
        }
        $owner = $this->username;
        $IBSng_uid = $this->createNewUser($group_name, $credit);
        if (!$IBSng_uid['result']) {
            return $IBSng_uid;
        }
        $IBSng_uid = $IBSng_uid['userId'];
        $action = 'IBSng/admin/plugins/edit.php?edit_user=1&user_id=' . $IBSng_uid . '&submit_form=1&add=1&count=1&credit=1&owner_name=' . $owner . '&group_name=' . $group_name . '&x=35&y=1&edit__normal_username=normal_username';
        $post_data = array_merge($this->IBSngUtils->userTargetPostData($IBSng_uid), $this->IBSngUtils->updateUserPasswordPostData($username, $password));
        $post_data['normal_save_user_add'] = 't';
        $post_data['credit'] = $credit;
        if (!is_null($multiLogin)) {
            $post_data['attr_update_method_1'] = "multiLogin";
            $post_data['has_multi_login'] = 't';
            $post_data['multi_login'] = $multiLogin;
        }
        $output = $this->request($action, $post_data, TRUE);
        if (strpos($output, 'exist')) {
            $this->_delUser($IBSng_uid, TRUE);
            return ['result' => FALSE, 'error' => "username already exists"];
        }
        if (strpos($output, 'IBSng/admin/user/user_info.php?user_id_multi')) {
            return ['result' => TRUE];
        }
    }

    protected function _delUser($username, $isUid = FALSE, $logs = TRUE, $audit = TRUE)
    {
        if ($isUid) {
            $uid = $this->getUserInfo($username, uid: TRUE)['info']['uid'] ?? NULL;
        }
        else {
            $uid = $this->getUserInfo($username)['info']['uid'] ?? NULL;
        }
        if (is_null($uid)) {
            return ['result' => FALSE, 'error' => "User not found"];
        }
        else {
            $action = 'IBSng/admin/user/del_user.php';
            $post_data['user_id'] = $uid;
            $post_data['delete'] = 1;
            $post_data['delete_comment'] = '';
            if ($logs)
                $post_data['delete_connection_logs'] = 'on';
            if ($audit)
                $post_data['delete_audit_logs'] = 'on';
            $output = $this->request($action, $post_data, TRUE);
            if (strpos($output, 'Successfully')) {
                return ['result' => TRUE];
            }
            else {
                return ['result' => FALSE, "error" => "Unknown"];
            }
        }
    }

    protected function getUserInfo($username, $withPassword = FALSE, $uid = FALSE, $output = NULL): array
    {
        $action = $uid
            ? 'IBSng/admin/user/user_info.php?user_id_multi=' . $username
            : 'IBSng/admin/user/user_info.php?normal_username_multi=' . $username;

        if (!$output) {
            $output = $this->request($action);
        }
        $start = microtime(TRUE);
        if (str_contains($output, 'does not exists')) {
            return ['result' => FALSE, 'error' => "[$username] Not Found"];
        }
        $dom = new DomDocument();
        @$dom->loadHTML($output);
        $finder = new DomXPath($dom);
        $getNodeValue = function ($class, $index = 0) use ($finder) {
            $nodes = $finder->query("//*[contains(@class, '$class')]");
            return trim($nodes->item($index)->nodeValue ?? '0');
        };
//        $locked = str_contains($getNodeValue('Form_Content_Row_Right_textarea_td_light'), 'Yes') ? '1' : '0';
        $userMulti = !str_contains($getNodeValue('Form_Content_Row_Right_userinfo_light', 4), 'instances') ? 0 : trim(str_replace('instances', '', $getNodeValue('Form_Content_Row_Right_userinfo_light', 4)));
        $groupMulti = !str_contains($getNodeValue('Form_Content_Row_groupinfo_light', 4), 'instances') ? 0 : trim(str_replace('instances', '', $getNodeValue('Form_Content_Row_groupinfo_light', 4)));

        $groupNode = $finder->query("//a[contains(@href, '/IBSng/admin/group/group_info.php?group_name=')]");
        $groupName = trim($groupNode->item(0)->nodeValue) ?? '0';

        if (str_starts_with($groupName, 'Server')) {
            throw new Exception("failed to retrieve group name");
        }

        $uid = $this->IBSngUtils->extractBetween($output, 'User ID', '<td class="Form_Content_Row_Right_light">', '</td>');
//        $owner = $this->IBSngUtils->extractBetween($output, 'Owner Admin', '<td class="Form_Content_Row_Right_dark">', '</td>');
//        $comment = $this->IBSngUtils->extractBetween($output, ' Comment :', '<td class="Form_Content_Row_Right_textarea_td_dark">', '</td>');
//        $name = $this->IBSngUtils->extractBetween($output, ' Name :', '<td class="Form_Content_Row_Right_textarea_td_light">', '</td>');
//        $phone = $this->IBSngUtils->extractBetween($output, ' Phone :', '<td class="Form_Content_Row_Right_textarea_td_dark">', '</td>');
        $creationDate = $this->IBSngUtils->extractBetween($output, 'Creation Date', '<td class="Form_Content_Row_Right_light">', '</td>');
//        $status = $this->IBSngUtils->extractBetween($output, 'Status', '<td class="Form_Content_Row_Right_dark">', '</td>');
        $exp = $this->IBSngUtils->extractBetween($output, 'Nearest Expiration Date:', '<td class="Form_Content_Row_Right_userinfo_light">', '</td>');
        $absExp = $this->IBSngUtils->extractBetween($output, 'Nearest Expiration Date:', '<td class="Form_Content_Row_Right_userinfo_light">', '</td>');
        $relExp = $this->IBSngUtils->extractBetween($output, 'Relative Expiration Date:', '<td class="Form_Content_Row_Right_userinfo_dark">', '</td>');
        $relExpGp = $this->IBSngUtils->extractBetween($output, '<div id="tab1_Exp_Dates_content" align=center>', '<td class="Form_Content_Row_groupinfo_dark" align=center>', '</td>');
        $firstLogin = $this->IBSngUtils->extractBetween($output, 'First Login:', '<td class="Form_Content_Row_Right_userinfo_dark">', '</td>');
        $credit = str_replace(',', '', $this->IBSngUtils->extractBetween($output, '<td class="Form_Content_Row_Left_dark"> 	Credit', '<td class="Form_Content_Row_Right_dark">', '<a class'));

        $firstLogin = str_starts_with($firstLogin, "-") ? '0' : $firstLogin;
        $exp = str_starts_with($exp, "-") ? '0' : $exp;
        $absExp = str_starts_with($absExp, "-") ? '0' : $absExp;
        $relExp = str_starts_with($relExp, "-") ? '0' : $relExp;
        $relExpGp = str_starts_with($relExpGp, "-") ? '0' : $relExpGp;

        if ($exp != 0) {
            $remainDays = round((strtotime($exp) - time()) / (3600 * 24));
        }
        else {
            if ($absExp != 0) {
                $expire = strtotime($absExp);
            }
            else if ($relExp != 0) {
                $expire = strtotime($relExp);
            }
            else if ($relExpGp != 0) {
                $expire = strtotime($relExpGp);
            }
            $remainDays = round(($expire - time()) / (3600 * 24));
        }

        $info = [
            'username' => $username,
//            'name' => $name ?: '0',
//            'comment' => $comment ?: '0',
//            'phone' => $phone ?: '0',
//            'owner' => $owner,
            'credit' => $credit,
            'uid' => $uid,
            'group' => $groupName,
            'first_login' => $firstLogin,
            'creation_date' => $creationDate,
            'nearest_expire_date' => $exp,
            'absolute_expire_date' => $absExp,
            'relative_expire_date' => $relExp,
            'relative_expire_date_gp' => $relExpGp,
            'remain_days' => $remainDays,
//            'status' => $status,
//            'locked' => $locked,
            'user_multi' => $userMulti,
            'group_multi' => $groupMulti
        ];
        if ($withPassword) {
            $info['password'] = $this->getPassword($uid)['password'];
        }
        return ['result' => TRUE, 'info' => $info];
    }


    protected function getPassword($uid): array|string
    {
        $action = 'IBSng/admin/plugins/edit.php';
        $postData['user_id'] = $uid;
        $postData['edit_user'] = 1;
        $postData['attr_edit_checkbox_2'] = 'normal_username';

        $output = $this->request($action, $postData);

        $phrase = '<td class="Form_Content_Row_Right_light">	<input type=text id="password" name="password" value="';
        $pos1 = strpos($output, $phrase);
        $leftover = str_replace($phrase, '', substr($output, $pos1, strlen($phrase) + 1000));
        $password = substr($leftover, 0, strpos($leftover, '"'));
        if (isset($password)) {
            return ['result' => TRUE, 'password' => trim($password)];
        }
        else {
            return ['result' => FALSE];
        }
    }

    private function createNewUser($group_name, $credit)
    {
        $action = 'IBSng/admin/user/add_new_users.php';
        $post_data['submit_form'] = 1;
        $post_data['add'] = 1;
        $post_data['count'] = 1;
        $post_data['credit'] = $credit;
        $post_data['owner_name'] = $this->username;
        $post_data['group_name'] = $group_name;
        $post_data['edit__normal_username'] = 1;
        $output = $this->request($action, $post_data, TRUE);
        if (str_contains($output, "Group name " . $group_name . " is invalid")) {
            return ["result" => FALSE, "error" => "group name is invalid"];
        }
        $pattern1 = 'user_id=';
        $pos1 = strpos($output, $pattern1);
        $sub1 = substr($output, $pos1 + strlen($pattern1), 100);
        $pattern2 = '&su';
        $pos2 = strpos($sub1, $pattern2);
        return ['result' => TRUE, 'userId' => substr($sub1, 0, $pos2)];
    }

    private function _resetFirstLogin($userName)
    {
        $userId = $this->getUserInfo($userName);
        if (!$userId['result']) {
            return ['result' => FALSE, 'error' => "User not found"];

        }
        $userId = $userId['info']['uid'];
        $action = "IBSng/admin/plugins/edit.php";
        $postData = $this->IBSngUtils->userTargetPostData($userId);
        $postData['edit_tpl_cs'] = "first_login";
        $postData['tab1_selected'] = "Exp_Dates";
        $postData['attr_update_method_0'] = "firstLogin";
        $postData['reset_first_login'] = "t";
        return ['result' => TRUE, 'data' => $this->request($action, $postData)];
    }

    public function getGroupsDetails($groupName = "")
    {
        if ($groupName != "") {
            $postData = $this->IBSngUtils->preparePostDataForGroup($groupName);
            try {
                $response = $this->request('IBSng/admin/plugins/edit.php', $postData);
                if (str_contains($response, "is invalid")) {
                    return ['status' => FALSE, 'error' => "Group name is invalid"];
                }
                $htmlSection = $this->IBSngUtils->extractHtmlSection($response, "<!-- Main Table -->", 'multi_login_select=new DomContainer();');
                $groupDetail = $this->IBSngUtils->parseGroupDetails($htmlSection);
                return ['status' => TRUE, 'data' => $groupDetail];
            } catch (Exception $e) {

            }
        }
        $action = "IBSng/admin/group/group_list.php";
        $out = $this->request($action);
        $groupsList = $this->IBSngUtils->extractHtmlSection($out, '<tr class="List_Head">', '<tr class="List_Foot_Line_red">');
        $groupNames = $this->IBSngUtils->extractGroupNames($groupsList);

        $groupsDetails = [];

        foreach ($groupNames as $group) {
            $postData = $this->IBSngUtils->preparePostDataForGroup($group);
            try {
                $response = $this->request('IBSng/admin/plugins/edit.php', $postData);
                $htmlSection = $this->IBSngUtils->extractHtmlSection($response, "<!-- Main Table -->", 'multi_login_select=new DomContainer();');
                $groupDetail = $this->IBSngUtils->parseGroupDetails($htmlSection);
                if ($groupDetail['group_name'] == $groupName) {
                    return [$groupDetail];
                }
                else {
                    $groupsDetails[] = $groupDetail;
                }
            } catch (Exception $e) {

            }
        }
        return $groupsDetails;
    }

    public function getGroupsName()
    {
        $action = "IBSng/admin/group/group_list.php";
        $out = $this->request($action);
        $groupsList = $this->IBSngUtils->extractHtmlSection($out, '<tr class="List_Head">', '<tr class="List_Foot_Line_red">');
        return $this->IBSngUtils->extractGroupNames($groupsList);
    }

    private function _editUser($userName, $attribute, $value, $userPasswordEdit)
    {
        $action = "IBSng/admin/plugins/edit.php";
        if (!in_array($attribute, ['normalAttrs', 'relExpDate', 'multi_login', 'group_name'])) {
            return ['result' => FALSE, 'error' => 'edited attribute not allowed'];
        }
        $userInfo = $this->getUserInfo($userName, TRUE);
        if (!$userInfo['result']) {
            return ['result' => FALSE, 'error' => "User not found"];
        }
        $userId = $userInfo['info']['uid'];
        $postData = $this->IBSngUtils->userTargetPostData($userId);
        if ($attribute == "normalAttrs") {
            if ($userPasswordEdit == "userName") {
                $newUser = $this->getUserInfo($value);
                if ($newUser['result']) {
                    return ['result' => FALSE, 'error' => "New user is existed"];
                }
                if (!$this->IBSngUtils->validateUserName($value)) {
                    return ['result' => FALSE, 'error' => "New user is not valid"];
                }
                $postData = array_merge($postData, $this->IBSngUtils->updateUserPasswordPostData($value, $userInfo['info']['password'], $userName));
            }
            else if ($userPasswordEdit == "password") {
                $postData = array_merge($postData, $this->IBSngUtils->updateUserPasswordPostData($userName, $value, $userName));
            }
        }
        else if ($attribute == "relExpDate") {
            $postData = array_merge($postData, $this->IBSngUtils->relativeExpireDatePost($userId, $value));
        }
        else if ($attribute == "multi_login") {
            $postData = array_merge($postData, $this->IBSngUtils->updateUserMultiLogin($value));
        }
        else if ($attribute == "group_name") {
            $groupDetail = $this->getGroupsDetails($value);
            if (count(array_keys($groupDetail)) > 1) {
                return ['result' => FALSE, 'error' => "Group not found"];
            }
            $postData = array_merge($postData, $this->IBSngUtils->updateUserGroupPostData($value));
        }
        $this->request($action, $postData);
        return ['result' => TRUE];
    }

    private function _getAllUsersList()
    {
        $group = $this->getGroupsDetails()[0];
        $lastUserId = $this->createNewUser($group, 10);
        $this->_delUser($lastUserId, TRUE);
        $usersId = [];
        for ($i = 1; $i < $lastUserId; $i++) {
            $userInfo = $this->getUser($i, FALSE, TRUE);
            if ($userInfo['result']) {
                $usersId[] = $userInfo['info']['uid'];
            }
        }
        return $usersId;

    }

    private function _changeCredit($userId, $credit, $comment = "")
    {
        $action = "IBSng/admin/user/change_credit.php";
        $postData['user_id'] = $userId;
        $postData['credit'] = $credit;
        $postData['credit_comment'] = $comment;
        $rep = $this->request($action, $postData);
        return ['result' => TRUE];
    }

    private function _addDays($userName, $days)
    {
        $action = 'IBSng/admin/plugins/edit.php';
        $user = $this->getUser($userName)['info'];
        if ($user['first_login'] == '0') {
            if ($user['relative_expire_date'] != "0") {
                $userRelativeDays = $this->IBSngUtils->calculateRelativeExpireDate($user['relative_expire_date']);
            }
            else {
                $userRelativeDays = $this->IBSngUtils->calculateRelativeExpireDate($user['relative_expire_date_gp']);
            }
            $new = $days + $userRelativeDays;
        }
        else {
            $exp = $user['nearest_expire_date'];
            if (strtotime($exp) <= time()) {
                $new = $days;
            }
            else {
                $date1 = strtotime($exp);
                $remain = $date1 - time();
                $remainDays = round($remain / (3600 * 24));
                $new = $remainDays + $days;
            }
            $this->_resetFirstLogin($user['username']);
        }
        echo "NEW : " . $new . "\n";
        $post_data = $this->IBSngUtils->relativeExpireDatePost($user['uid'], $new);
        $output = $this->request($action, $post_data, TRUE);
        return ['result' => TRUE];
    }

    private function _addMultiLogin($userName, $multiLogin)
    {
        $user = $this->getUser($userName)['info'];
        $oldMultiLogin = $user['user_multi'];
        $newMultiLogin = $oldMultiLogin + $multiLogin;
        return $this->editUser($userName,"multi_login",$newMultiLogin,"");
    }
}
//const IBSngConnection = [
//    'username' => 'system',
//    'password' => '4087',
//    'hostname' => '65.108.148.49',
//];
//
//$ibs = new IBSngRequests(IBSngConnection);
//$ibs->addMultiLogin("1qbv36","1");
